
/*******************************************************************************/
/*  
	This script produces Table A1
	Mongey, Pilossoph, and Weinberg: "Which Workers Bear the Burden of Social Distancing?"
	Journal of Economic Inequality 2021	
    --
	Alexander Weinberg
	February 16, 2021
*/
/*******************************************************************************/

// __________________________________________________
// Adjust for inflation
import delimited "../Raw Data/CPIAUCSL.csv", clear 	// Download from FRED
gen year = substr(date, 1, 4)
destring year, replace

sort year 
replace cpi = cpi[_n-1] if mi(cpi) & year == 2020 	// note 2020 CPI unavailable

qui summ cpi if (year==2019)						// normalize to 2019 prices
local cpi2019 = r(mean)
replace cpi = cpi/`cpi2019'
keep year cpi 

// ___________________________________________________
// Load monthly CPS
merge 1:m year using "../Data/CPS/monthly_cps_2010_2020.dta"	// Merge in basic monthly CPS 2010-2020
drop if _merge == 1
keep if labforce == 2 // in labor force

// __________________________________________________
// Generate Binary Variables
gen married 		= marst == 1							// Married = spouse present
gen is_citizen      = (citizen != 5)

gen parttime 		= (uhrswork1 < 30) 						// Fulltime = 30 hr/week
replace parttime 	= . if (uhrswork1 == 0) | (uhrswork1 >= 997) 
gen fulltime 		= 1 - parttime
drop parttime 

gen male 			= (sex == 1)
gen white		    = (race == 100)
gen old 			= (age >= 50)

gen college 		= inrange(educ, 111, 125)
replace college 	= . if educ > 125

gen born_50states 	= (bpl == 9900)

// __________________________________________________
// Wages
replace earnweek  	= . if earnweek >= 9999
replace earnweek 	= earnweek / cpi

replace uhrswork1 	= . if uhrswork1 > 99
replace uhrsworkorg = . if uhrsworkorg > 99

gen fedminwage 		= 7.25
gen wage 			= earnweek / uhrswork1 
replace wage 		= . if wage < 0.5 * fedminwage 			// very low earners
replace wage 		= . if (earnweek>0) & (uhrswork1==0)

xtile inc_tile		= wage if !mi(wage), nq(2) 
gen bottom50		= (inc_tile  == 1) if !mi(inc_tile) 

// ___________________________________________________
// LABEL VARS
label var college "College degree"
label var male "Male"
label var born_50states "Born in US"
label var married "Married"
label var is_citizen "US citizen"
label var white "White"
label var fulltime "Full-time employed"
label var old "Age 50+"

label var age "Age"
label var wage "Hourly wage*"
label var uhrswork1 "Usual weekly hours*"
label var years_school "Years of education"

// ___________________________________________________
// SUMMARY STATS
eststo clear
eststo: qui estpost summarize ///
    age wage uhrswork1 years_school college male born_50states married ///
	is_citizen white fulltime old if month==2 [w=compwt], detail // format

eststo: qui estpost summarize ///
    age wage uhrswork1 years_school college male born_50states married ///
	is_citizen white fulltime old if month==4 [w=compwt], detail // format

eststo: qui estpost summarize ///
    age wage uhrswork1 years_school college male born_50states married ///
	is_citizen white fulltime old if month==8 [w=compwt], detail // format

esttab using "../Tables/tableA1.tex", cells("mean(fmt(2)) sd(fmt(2))") label nodepvar  ///
	varwidth(25) nonumbers booktabs title("CPS - Monthly survey") ///
	replace mtitles("Feb" "April" "August")

// end
